<?php

namespace Mnv\Modules\User;

use Mnv\Core\Auth\Errors\MissingCallbackError;
use Mnv\Core\Config;
use Mnv\Core\Locale\I18N;
use Mnv\Models\Users\UserTypes;
use Mnv\Modules\SmsGateway;

use Mnv\Core\Auth\Errors\AuthError;
use Mnv\Core\Auth\Errors\UserPhoneRequiredError;
use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Core\Auth\Exceptions\NotLoggedInException;
use Mnv\Core\Auth\Exceptions\PhoneNotPaidException;
use Mnv\Core\Auth\Exceptions\EmailNotPaidException;
use Mnv\Core\Auth\Exceptions\InvalidPhoneException;
use Mnv\Core\Auth\Exceptions\TokenExpiredException;
use Mnv\Core\Auth\Exceptions\InvalidEmailException;
use Mnv\Core\Auth\Exceptions\ResetDisabledException;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\UnknownUsernameException;
use Mnv\Core\Auth\Exceptions\PhoneNotVerifiedException;
use Mnv\Core\Auth\Exceptions\ValidatePasswordException;
use Mnv\Core\Auth\Exceptions\AttemptCancelledException;
use Mnv\Core\Auth\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Auth\Exceptions\UserAlreadyExistsException;
use Mnv\Core\Auth\Exceptions\AmbiguousUsernameException;
use Mnv\Core\Auth\Exceptions\DuplicateUsernameException;
use Mnv\Core\Auth\Exceptions\InvalidSelectorTokenPairException;
use Mnv\Core\Database\Throwable\IntegrityConstraintViolationException;

class Phone extends User implements UserInterface
{
    /** @var SmsGateway */
    protected SmsGateway $sms;

    public $action;
    public array $user = [];
    public $uploadFile;
    public $address = [];

    /**
     * Phone constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->sms = new SmsGateway(Config::getValue('sms_url'),Config::getValue('sms_login'), Config::getValue('sms_password'));
    }

    public function run(): void
    {
        switch ($this->action) {
            case 'auth':
                $this->auth();
                break;
            case 'registration':
                $this->registration();
                break;
            /** Подтверждение регистрации и авторизации через смс */
            case 'confirm':
                $this->confirmSms();
                break;
            case 'changeUser':
                $this->user['address'] = null;
                if (!empty($this->address)) {
                    $this->user['address'] = json_encode($this->address, JSON_UNESCAPED_UNICODE);
                }
                $this->changeUser($this->user);
                break;
            case 'changeContact':
                $this->changePhone();
                break;
            case 'repeatCode':
                $this->repeatCode();
                break;
            case 'recoveryPhone':
                $this->recoveryPhone();
                break;
            case 'recoveryConfirm':
                $this->recoveryConfirmPhone();
                break;
            default:
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> 'Error: method not found');
                break;
        }
    }

    /** Авторизация */
    public function auth()
    {
        global $userAuth;

        try {
            $rememberDuration = (isset($this->user['remember']) && $this->user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;
            $this->user['phone'] = $this->sms->formatPhoneNumber($this->user['phone']);
            if (Config::getValue('twoFactorAuth') == 1) {
                /** подтверждение через смс шлюз */
                $callback = function ($phone, $tokenSms) {
                    $phone = $this->sms->formatPhoneNumber($phone);
                    if ($this->sms->sendSms($phone, I18N::locale("Vash kod podtverjdeniya na sayte Brend Goda 2023 : $tokenSms", "Sizning tasdiqlash kodingiz 2023 yilgi brend veb-saytida : $tokenSms", "Your confirmation code on the 2023 Brand website : $tokenSms")))  {
                        $phoneHide = substr_replace($phone,'****',4,13);
                        $this->response = array('status' => 200, 'sms' => true, 'type'=> 'success',
                            'message' => I18N::locale("На Ваш номер $phoneHide телефона отправлен код подтверждения!", "$phone telefon raqamingizga tasdiqlash kodi!", "Confirmation code to your phone number $phone!")
                        );
                    } else {
                        $this->response = array('status' => 403, 'type'=> 'error',
                            'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                        );
                    }
                };
            } else {
                $callback = null;
                $this->response = array('status' => 200, 'type'=> 'success',
                    'message' => I18N::locale("Добро пожаловать!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
                );
            }

            $userAuth->loginWithPhone($this->user['phone'], $this->user['password'], (int)Config::getValue('twoFactorAuth'), $rememberDuration, $callback);
        }
        catch (UserPhoneRequiredError $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale("Поле номер телефона не должен быть пустым", "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan", 'The user with this phone number is not registered')
            );
        }
        catch (InvalidPhoneException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale("Пользователь с таким номером телефона не зарегистрирован", "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan", 'The user with this phone number is not registered')
            );
        }
        catch (ValidatePasswordException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale("Пароль не должен быть пустым или кол-во символов не должно быть меньше 3", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (InvalidPasswordException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale("Вы указали не верный пароль", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (UnknownIdException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale("Пользователь не найден", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (PhoneNotVerifiedException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale('Номер телефона не был подтвержден', 'Email tasdiqlanmagan', 'Email not verified')
            );
        }
        catch (TooManyRequestsException $e) {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale("Слишком много запросов", "Juda ko'p so'rovlar", 'Too many requests')
            );
        } catch (AuthError $e) {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale("Ошибка авторизации", "Avtorizatsiya xatosi", "Authorisation Error")
            );
        } catch (AttemptCancelledException $e) {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale('Попытка отменена', 'Попытка отменена', 'Попытка отменена')
            );
        } catch (IntegrityConstraintViolationException $e) {
            $this->response = array('status' => 403, 'type'=> 'error',
                'message' => I18N::locale('Ваш ip заблокирован из-за нарушений авторизации', 'Email tasdiqlanmagan', 'Email not verified')
            );
        }

    }

    public function registration(): void
    {
        global $userAuth;

        if (!empty($this->user)) {
            if (!empty($this->user['phone'])) {
                $this->user['phone'] = $this->sms->formatPhoneNumber($this->user['phone']);

                try {
                    if ((int)Config::getValue('twoFactorAuth') == 1) {
                        $callback = function ($phone, $tokenSms) {  /** подтверждение через смс шлюз */
                            $phone = $this->sms->formatPhoneNumber($phone);
                            if ($this->sms->sendSms($phone, I18N::locale("Vash kod podtverjdeniya na sayte Brend Goda 2023 : $tokenSms", "Sizning tasdiqlash kodingiz 2023 yilgi brend veb-saytida : $tokenSms", "Your confirmation code on the 2023 Brand website : $tokenSms")))  {
                                $phoneHide = substr_replace($phone,'****',4,13);
                                $this->response = array('status' => 200, 'sms' => true, 'type'=> 'success', 'message' => I18N::locale("На Ваш номер $phoneHide телефона отправлен код подтверждения!", "$phoneHide telefon raqamingizga tasdiqlash kodi!", "Confirmation code to your phone number $phone!")
                                );
                            } else {
                                $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!"));
                            }
                        };
                    } else {
                        $this->response = array('status' => 200, 'type'=> 'success',
                            'message' => I18N::locale("Добро пожаловать!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
                        );
                        $callback = null;
                    }

                    $userAuth->registerWithUniqueUsername('phone', $this->user, (int)Config::getValue('twoFactorAuth'), $callback);

                }  catch (InvalidEmailException $e) {
                    $this->response =  array(
                        'status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale("Неверный номер телефона", "Yaroqsiz elektron pochta manzili", "Invalid email address")
                    );
                }
                catch (ValidatePasswordException $e) {
                    $this->response = array('status' => 403, 'type'=> 'error',
                        'message' => I18N::locale("Пароль не должен быть пустым или кол-во символов не должно быть меньше 3", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
                    );
                } catch (InvalidPasswordException $e) {
                    $this->response =  array(
                        'status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale("Invalid password", "Invalid password", "Invalid password")
                    );
                } catch (UserAlreadyExistsException $e) {
                    $this->response =  array(
                        'status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale("Пользователь уже существует", "Foydalanuvchi allaqachon mavjud", "User already exists")
                    );
                } catch (DuplicateUsernameException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale("Пользователь с таким номером телефона уже зарегистрирован", "Foydalanuvchi allaqachon mavjud", "User already exists")
                    );
                } catch (TooManyRequestsException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale("Слишком много запросов", "Juda ko'p so'rovlar", "Too many requests")
                    );
                } catch (AuthError $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error', 'message'=> $e->getMessage());
                }
            } else {
                $this->response =  array(
                    'status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Поле email не должно быть пустым!", "'Elektron pochta maydoni bo'sh bo'lmasligi kerak!'", "The email field must not be empty!")
                );
            }
        } else {
            $this->response = array(
                'status' => 403, 'type'=> 'error',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar boʻsh!", "Sent data is empty!"),
            );
        }

    }

    /**
     * Подтверждение регистрации и авторизации через смс
     */
    public function confirmSms(): void
    {
        global $userAuth;

        if (!empty($this->user['code'])) {
            try {
                $userAuth->confirmPhoneAndSignIn($this->user['code']);
                $this->response =  array('status'=> 200, 'type'=> 'success', 'redirect' => SITE_URL . '/add-application/',
                    'message'=> I18N::locale("Добро пожаловать!", "Xush kelibsiz!", "Welcome!")
                );

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Не верный код подтверждения!", "Tasdiq kodi noto'g'ri!", "Invalid confirmation code!")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale('Слишком много запросов', 'Juda koʻp soʻrovlar', 'Too many requests')
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Пользователь c таким номером телефона уже существует", "Bu telefon raqamiga ega foydalanuvchi allaqachon mavjud", 'A user with this phone number already exists')
                );
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Ошибка авторизации", "Avtorizatsiya xatosi", "Authorization error")
                );
            }
        } else {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    /**
     * Восстановление пароля
     */
    public function recoveryPhone(): void
    {
        global $userAuth;

        if (!empty($this->user) && !empty($this->user['phone'])) {
            $this->user['phone'] = $this->sms->formatPhoneNumber($this->user['phone']);
            if ((int)Config::getValue('twoFactorAuth') == 1) {
                try {
                    $userAuth->forgotPhonePassword($this->user['phone'], function ($phone, $tokenSms) {
                        $phone = $this->sms->formatPhoneNumber($phone);
                        if ($this->sms->sendSms($phone, I18N::locale("Ваш код подтверждения на сайте " . GLOBAL_URL . " : $tokenSms", "Saytdagi tasdiqlash kodingiz " . GLOBAL_URL . " : $tokenSms", "Your confirmation code on the website" . GLOBAL_URL . " : $tokenSms"))) {
                            $phoneHide = substr_replace($phone,'****',4,13);
                            $this->response = array('status' => 200, 'sms' => true, 'type'=> 'success',
                                'message' => I18N::locale("На Ваш номер $phoneHide телефона отправлен код подтверждения!", "$phoneHide telefon raqamingizga tasdiqlash kodi!", "Confirmation code to your phone number $phone!")
                            );
                        } else {
                            $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!"));
                        }
                    });
                }
                catch (PhoneNotPaidException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale('Не была произведена оплата', 'Ne byla prizvedena oplata', 'Payment has not been made')
                    );
                }
                catch (PhoneNotVerifiedException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale('Номер телефона не подтвержден', 'Telefon raqami tasdiqlanmagan', 'Phone number not verified')
                    );
                }
                catch (ResetDisabledException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale('Сброс пароля отключен', 'Parolni tiklash o\'chirilgan', 'Password reset is disabled')
                    );
                }
                catch (TooManyRequestsException $e) {
                    $this->response =  array('status'=> 403, 'type'=> 'error',
                        'message'=> I18N::locale('Слишком много запросов', 'Juda koʻp soʻrovlar', 'Too many requests')
                    );
                } catch (AuthError $e) {
                }
            }

        } else {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale('Отправленные данные пусты!', 'Yuborilgan maʼlumotlar boʻsh!', 'Sent data is empty!')
            );
        }
    }

    /** Подтверждение восстановление пароля */
    public function recoveryConfirmPhone(): void
    {
        global $userAuth;

        if (!empty($this->user['code'])) {
            try {
                $phoneBeforeAndAfter = $userAuth->confirmPhone($this->user['code']);
                if ($phoneBeforeAndAfter[1] !== null) {
                    $userAuth->recoveryPhoneGenerationPassword($phoneBeforeAndAfter[1], function ($password, $phone) {
                        $phone = $this->sms->formatPhoneNumber($phone);
                        if ($this->sms->sendSms($phone, "Ваш новый пароль для входа на сайт: $password")) {
                            $phoneHide = substr_replace($phone,'****',4,13);
                            $this->response = array('status' => 200, 'sms' => true, 'type'=> 'success',
                                'message' => I18N::locale("На Ваш номер $phoneHide телефона отправлен новый пароль!", "$phone telefon raqamingizga tasdiqlash kodi!", "Confirmation code to your phone number $phone!")
                            );
                        } else {
                            $this->response = array('status' => 403, 'type'=> 'error',
                                'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                            );
                        }
                    });
                }
                $this->response =  array('status'=> 200, 'type'=> 'success', 'redirect' => SITE_URL . '/sign-in/',
                    'message'=> I18N::locale("На Ваш номер телефона выслан новый пароль для входа!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
                );
            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Не верный код!", "Kod noto'g'ri!", "Incorrect code!")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Слишком много запросов", "Juda ko'p soʻrovlar", "Too many requests")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Пользователь уже существует", "Foydalanuvchi allaqachon mavjud", "User already exists")
                );
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Ошибка авторизации", "Foydalanuvchi allaqachon mavjud", "User already exists",)
                );
            } catch (EmailNotVerifiedException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Ошибка авторизации", "Foydalanuvchi allaqachon mavjud", "User already exists")
                );
            } catch (InvalidEmailException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Ошибка авторизации", "Foydalanuvchi allaqachon mavjud", "User already exists")
                );
            } catch (ResetDisabledException $e) {
                $this->response =  array('status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Ошибка авторизации", "Foydalanuvchi allaqachon mavjud", "User already exists")
                );
            }
        } else {
            $this->response =  array('status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }

    }

    public function changePhone(): void
    {
        global $userAuth;

        if (!empty($this->user)) {
            $this->user['phone'] = $this->sms->formatPhoneNumber($this->user['phone']);
            try {
                $userAuth->changeUserPhone($this->user);
                $this->response =  array(
                    'status'=> 200, 'type'=> 'success',
                    'message'=> I18N::locale("Ваши данные успешно изменены", "Password has been changed", "Password has been changed")
                );
            }  catch (NotLoggedInException $e) {
                $this->response =  array(
                    'status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Not logged in", "Not logged in", "Not logged in")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  [
                    'status'=> 403, 'type'=> 'error',
                    'message'=> I18N::locale("Слишком много запросов!", "Too many requests!", "Too many requests!")
                ];
            }
        } else {
            $this->response =  array(
                'status'=> 403, 'type'=> 'error',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }


    /** не получил код */
    private function repeatCode(): void
    {
        global $userAuth;

        if (!empty($this->user['phone'])) {
            $rememberDuration = (isset($this->user['remember']) && $this->user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;

            if ((int)Config::getValue('twoFactorAuth') == 1) {
                $callback = function ($phone, $tokenSms) {  /** подтверждение через смс шлюз */
                    $phone = $this->sms->formatPhoneNumber($phone);
                    if ($this->sms->sendSms($phone, I18N::locale("Ваш код подтверждения на сайте " . GLOBAL_URL . " : $tokenSms", "Saytdagi tasdiqlash kodingiz " . GLOBAL_URL . " : $tokenSms", "Your confirmation code on the website" . GLOBAL_URL . " : $tokenSms")))  {
                        $phoneHide = substr_replace($phone,'****',4,13);
                        $this->response = array('status' => 200, 'sms' => true, 'type' => 'success', 'phone' => $phone, 'message' => I18N::locale(
                            "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                            "$phoneHide telefon raqamingizga tasdiqlash kodi!",
                            "Confirmation code to your phone number $phoneHide!"
                        ));
                    } else {
                        $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!"));
                    }
                };
            } else {
                $callback = null;
            }

            try {
                $userAuth->createRepeatPhoneCode($this->user['phone'], $rememberDuration, $callback);
            } catch (MissingCallbackError $e) {
                $this->response = array('status' => 403, 'type'=> 'error',
                    'message' => I18N::locale("Ошибка callback", "Error callback", 'Error callback')
                );
            } catch (InvalidPhoneException $e) {
                $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale(
                    "Пользователь с таким номером телефона не зарегистрирован",
                    "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan",
                    "The user with this phone number is not registered"
                ));
            }
        }

    }
}